<?php 
/**
* Extension of the Codeigniter {@link http://codeigniter.com/user_guide/helpers/inflector_helper.html inflector helper}.
*
* @package direct-project-innovation-initiative
* @subpackage helpers
* @filesource
*/

#load_libraries('Error', 'Validator'); //autoloaded in this application, don't add the overhead of an attempted re-load

/**
* Returns 'a' or 'an' depending on whether or not the next word is a vowel.
* @see http://en.wikipedia.org/wiki/Indefinite_article#Indefinite_article
* @param string The word that will follow the indefinite article.
* @return string 'a' or 'an'
*/
function indefinite_article($next_word){
	if(empty($next_word) || !is_string($next_word))
		return false;
	
	$next_word = mb_strtolower($next_word);	
		
	$vowel_rule_exceptions = array('one', 'union', 'united', 'user'); 	
	foreach($vowel_rule_exceptions as $vowel_rule_exception){
		if(string_begins_with($vowel_rule_exception, $next_word))
			return 'a';
	}
			
	foreach(mb_str_split($next_word) as $char){ 
		if(is_numeric($char)){
			return 'a';
		}elseif(is_vowel($char)){ //checks for vowels, with or without diacritic marks
			return 'an';
		}elseif(preg_match('/\p{L}/u', $char)){ //any letter
			return 'a';
		}
	}
	return 'a'; //string didn't have any letters in it, just punctuation or numbers
}

/**
* Returns true if the given char is a vowel, with or without a diacritical mark
* Note that this may not be comprehensive - we may still need to add more vowels with diacritical marks
* @param string
* @return boolean
*/
function is_vowel($char){
	if(empty($char)) return false;
	if(!is_string($char)) return should_be('string');
	if(!preg_match('/\p{L}/u', $char)) return false; //return false if it's not a letter at all
	return preg_match('/[aàáâãäåæeèéêëiìíîïoòóôõöuùúûü]/u', mb_strtolower($char));
}


/**
* Overrides the CI humanize function to autmatically capitalize some common acronyms.
* @see http://codeigniter.com/user_guide/helpers/inflector_helper.html
* @param string The text to be humanized.
* @param string 
*/
function humanize($string){
	$string = mb_convert_case(preg_replace('/[_]+/', ' ', mb_strtolower(trim($string))), MB_CASE_TITLE);
	
	$acronyms = array( 'http', 'icn', 'id', 'ssn', 'url', 'va', 'vler');
	foreach($acronyms as $acronym){		
		if(string_contains(ucfirst($acronym), $string)){
			$pattern = '/(\b)'.ucfirst($acronym).'(\b)/';
			$replacement = '$1'.mb_strtoupper($acronym).'$2';
			$string = preg_replace($pattern, $replacement, $string);
		}
	}
	return $string;
}

/**
* Returns either "he" or "she" as appropriate.
* If the gender cannot be determined or is not supplied, will default to "they."
* @param string Gender (m/f/male/female/neutral)
* @return string
*/
function third_person_subjective_pronoun($gender = 'neutral'){
	if(!validates_as('string', $gender)) return should_be('string', $gender);
	$gender = mb_strtolower($gender);
	$pronoun_map = array('m' => 'he', 'f'=> 'she', 'male' => 'he', 'female'=>'she', 'neutral' => 'they');
	if(!array_key_exists($gender, $pronoun_map)) return should_be('"male", "female", or "neutral"', $gender);
	return $pronoun_map[$gender];
}	


/**
* Returns either "her" or "his" as appropriate.
* If the gender cannot be determined or is not supplied, will default to "their."
* @param string Gender (m/f/male/female/neutral)
* @return string
*/
function third_person_possessive_pronoun($gender = 'neutral'){
	if(!validates_as('string', $gender)) return should_be('string', $gender);
	$gender = mb_strtolower($gender);
	$pronoun_map = array('m' => 'his', 'f'=> 'her', 'male' => 'his', 'female'=>'her', 'neutral' => 'their');
	if(!array_key_exists($gender, $pronoun_map)) return should_be('"male", "female", or "neutral"', $gender);
	return $pronoun_map[$gender];
}


/**
* Returns either "him" or "her" as appropriate.
* If the gender cannot be determined or is not supplied, will default to "them."
* @param string Gender (m/f/male/female/neutral)
* @return string
*/
function third_person_object_pronoun($gender = 'neutral'){
		if(!validates_as('string', $gender)) return should_be('string', $gender);
	$gender = mb_strtolower($gender);
	$pronoun_map = array('m' => 'him', 'f'=> 'her', 'male' => 'him', 'female'=>'her', 'neutral' => 'them');
		if(!array_key_exists($gender, $pronoun_map)) return should_be('"male", "female", or "neutral"', $gender);
	return $pronoun_map[$gender];
}	



/**
* Returns either the singular or plural version of a word based on the specified quantity.
* Note that this function uses the standard {@link plural} and {@link singular} functions, and can only plural/singularlize as intelligently as they do.
* @param string The word that may need to be made plural or singular.
* @param int The number of things that we
*/
function pluralize_if_necessary($word, $quantity){
	if(!validates_as('string', $word)) return should_be('string', $word);
	if(!validates_as('numeric', $quantity)) return should_be('numeric', $quantity);
	
	if(empty($word)) return $word;
	
	//CI's plural and singular functions mess with case for some reason, so let's make sure that a word that begins with an upper or lower case remains the same
	//if we were really cool, we would worry about casing after the first letter as well, but this is the most common issue, so let's go with this for now.
	
	$first_char = mb_substr($word, 0, 1);
	$should_be_capitalized = ($first_char == mb_strtoupper($first_char));		
	
	if($quantity > 1)	
		$word = plural($word);
	else
		$word = singular($word);
	
	if($should_be_capitalized){
		$word = mb_strtoupper(mb_substr($word, 0, 1)).mb_substr($word, 1);
	}
	
	return $word;
}

/**
* Converts small numbers to their word equivalent.
* In accordance with the MLA and Chicago style guides, large numbers and decimals are not converted.  Theoretically, "large" should be 
* "above 100", but thus far we have been too lazy to code that.  Also, it's generally advised to be consistent within a sentence/visual region of text,
* so you may want to stick with numbers if it seems likely that one of the numbers in a sentence will be formatted differently than the others.
* @param number
* @param string|number
*/
function number_as_text($number){
	if(!validates_as('numeric', $number)) return should_be('numeric', $number);
	if(!validates_as('integer', $number) || $number > 10 || $number < 0 ) return $number; //decimals don't get spelled out
	$numbers_as_text = array('zero', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'ten');
	return element($number, $numbers_as_text, $number);
}

/**
* List the items of an array as a standard-English list.
* 
* For example, array(a, b, c) would become "a, b, and c".  Note that this list makes use of the serial/Oxford/Harvard comma, which is not considered correct 
* in all style guides.
*
* "There are people who embrace the Oxford comma, and people who don't, and I'll just say this: never get between these people when drink has been taken." --     e Truss
* @see http://en.wikipedia.org/wiki/Serial_comma
*
* @param array A list of items that can easily be converted to strings (e.g., numbers or strings)
* @param string
*/
function array_to_human_readable_list($array=array(), $prefix_for_last_item = 'and '){
	if(!validates_as('array', $array)) return should_be('array', $array);
	if(!validates_as('string', $prefix_for_last_item)) return should_be('string', $prefix_for_last_item);
	
	if(empty($array)) return '';
	if(count($array) == 1) return first_element($array);
	if(count($array) == 2) return implode(' '.$prefix_for_last_item.' ', $array);
	$array[array_last_key($array)] = $prefix_for_last_item.' '.last_element($array);
	return implode(', ', $array);
}

function boolean_to_english($boolean){
	if($boolean)
		return 'True';
	return 'False';	
}

function add_dashes_to_ssn($ssn_without_dashes){
	if(!validates_as('numeric', $ssn_without_dashes) || strlen($ssn_without_dashes) != 9) return should_be('social security number without dashes', $ssn_without_dashes);
	return substr($ssn_without_dashes, 0, 3).'-'.substr($ssn_without_dashes, 3, 2).'-'.substr($ssn_without_dashes, 5, 4);
}

/* End of file inflector_helper.php */